<?php

namespace App\Filament\Resources\LancamentoContabils\Tables;

use App\Models\LancamentoContabil;
use Carbon\Carbon;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\Filter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use App\Models\CentroCusto;

class LancamentoContabilsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(function (Builder $query) {
                // Padrão: dados do mês corrente
                $query->whereMonth('data_emissao', now()->month)
                    ->whereYear('data_emissao', now()->year);
            })
            ->columns([
                TextColumn::make('tipo')
                    ->searchable()
                    ->badge()
                    ->color(fn(string $state): string => match ($state) {
                        'entrada' => 'success',
                        'saida' => 'danger',
                        default => 'gray',
                    }),
                TextColumn::make('conta.banco')
                    ->searchable(),
                TextColumn::make('cliente.nome')
                    ->searchable(),
                TextColumn::make('fornecedor.nome')
                    ->searchable(),
                TextColumn::make('frete.numero_frete')
                    ->searchable(),
                TextColumn::make('factura.numero')
                    ->searchable(),
                TextColumn::make('conta_receber.numero')
                    ->searchable(),
                TextColumn::make('valor')
                    ->numeric()
                    ->sortable()
                    ->money('MZN')
                    ->summarize([
                        \Filament\Tables\Columns\Summarizers\Sum::make()
                            ->money('MZN')
                            ->label('Total Geral'),
                    ]),
                TextColumn::make('forma_pagamento')
                    ->searchable()
                    ->badge(),
                TextColumn::make('descricao')
                    ->searchable()
                    ->limit(50),
                TextColumn::make('centro_custo.nome')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('data_emissao')
                    ->date('d/m/Y')
                    ->sortable(),
                TextColumn::make('data_pagamento')
                    ->date('d/m/Y')
                    ->sortable()
                    ->toggleable(),
                TextColumn::make('observacoes')
                    ->limit(50)
                    ->toggleable(),
                TextColumn::make('created_at')
                    ->dateTime('d/m/Y H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('updated_at')
                    ->dateTime('d/m/Y H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('deleted_at')
                    ->dateTime('d/m/Y H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('contaPagarIva.valor')
                    ->label('IVA (16%)')
                    ->money('MZN')
                    ->formatStateUsing(fn($state) => $state ? number_format($state, 2, ',', '.') . ' MZN' : 'N/A'),

                TextColumn::make('contaPagarIva.status')
                    ->label('Status IVA')
                    ->badge()
                    ->color(fn($state) => match ($state) {
                        'pago' => 'success',
                        'pendente' => 'warning',
                        'cancelado' => 'danger',
                        default => 'gray',
                    }),

            ])
            ->filters([
                // NOVO FILTRO: Por Centro de Custo
                Filter::make('centro_custo')
                    ->form([
                        Select::make('centro_custo_id')
                            ->label('Centro de Custo')
                            ->placeholder('Selecione um centro de custo')
                            ->options(function () {
                                return CentroCusto::query()
                                    ->orderBy('nome')
                                    ->pluck('nome', 'id')
                                    ->toArray();
                            })
                            ->searchable()
                            ->preload(),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->when(
                            $data['centro_custo_id'] ?? null,
                            fn(Builder $query, $centroCustoId): Builder => $query->where('centro_custo_id', $centroCustoId),
                        );
                    })
                    ->indicateUsing(function (array $data): ?string {
                        if (!($data['centro_custo_id'] ?? null)) {
                            return null;
                        }

                        $centroCusto = CentroCusto::find($data['centro_custo_id']);
                        return $centroCusto ? 'Centro de Custo: ' . $centroCusto->nome : null;
                    }),

                // Filtro por período
                Filter::make('data_emissao')
                    ->form([
                        DatePicker::make('data_inicio')
                            ->label('Data Início')
                            ->default(now()->startOfMonth()),
                        DatePicker::make('data_fim')
                            ->label('Data Fim')
                            ->default(now()->endOfMonth()),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['data_inicio'],
                                fn(Builder $query, $date): Builder => $query->whereDate('data_emissao', '>=', $date),
                            )
                            ->when(
                                $data['data_fim'],
                                fn(Builder $query, $date): Builder => $query->whereDate('data_emissao', '<=', $date),
                            );
                    })
                    ->indicateUsing(function (array $data): array {
                        $indicators = [];

                        if ($data['data_inicio'] ?? null) {
                            $indicators[] = 'Data início: ' . Carbon::parse($data['data_inicio'])->format('d/m/Y');
                        }

                        if ($data['data_fim'] ?? null) {
                            $indicators[] = 'Data fim: ' . Carbon::parse($data['data_fim'])->format('d/m/Y');
                        }

                        return $indicators;
                    }),

                // Filtro rápido por mês
                SelectFilter::make('mes_referencia')
                    ->label('Mês de Referência')
                    ->options([
                        'current' => 'Mês Corrente',
                        'last' => 'Mês Passado',
                        'next' => 'Próximo Mês',
                    ])
                    ->default('current')
                    ->query(function (Builder $query, array $data) {
                        if (!isset($data['value']) || $data['value'] === 'current') {
                            return $query
                                ->whereMonth('data_emissao', now()->month)
                                ->whereYear('data_emissao', now()->year);
                        }

                        return match ($data['value']) {
                            'last' => $query
                                ->whereMonth('data_emissao', now()->subMonth()->month)
                                ->whereYear('data_emissao', now()->subMonth()->year),
                            'next' => $query
                                ->whereMonth('data_emissao', now()->addMonth()->month)
                                ->whereYear('data_emissao', now()->addMonth()->year),
                            default => $query,
                        };
                    }),

                // Filtro por ano
                SelectFilter::make('ano')
                    ->options(
                        collect(range(now()->year - 5, now()->year + 1))
                            ->mapWithKeys(fn($ano) => [$ano => $ano])
                            ->toArray()
                    )
                    ->default(now()->year)
                    ->query(function (Builder $query, array $data) {
                        return $query->when(
                            $data['value'] ?? null,
                            fn($q, $ano) =>
                            $q->whereYear('data_emissao', $ano)
                        );
                    }),

                // Filtro por tipo (entrada/saida)
                SelectFilter::make('tipo')
                    ->options([
                        'entrada' => 'Entrada (Receita)',
                        'saida' => 'Saída (Despesa)',
                    ])
                    ->multiple(),

                // Filtro por forma de pagamento
                SelectFilter::make('forma_pagamento')
                    ->options([
                        'cash' => 'Cash',
                        'transferencia' => 'Transferência',
                        'deposito bancario' => 'Depósito Bancário',
                        'cartao_credito' => 'Cartão de Crédito',
                        'cartao_debito' => 'Cartão de Débito',
                        'cheque' => 'Cheque'
                    ]),
                
                // Filtro por status
                SelectFilter::make('status')
                    ->options([
                        'pago' => 'Pago',
                        'pendente' => 'Pendente',
                        'cancelado' => 'Cancelado',
                    ])
                    ->multiple(),
            ])
            ->filtersLayout(FiltersLayout::AboveContent)
            ->defaultSort('data_emissao', 'desc')
            ->actions([
                ViewAction::make(),
                EditAction::make(),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ])
            ->searchable()
            ->persistSearchInSession()
            ->persistColumnSearchesInSession()
            ->persistFiltersInSession()
            ->header(function () {
                // Obter os totais filtrados
                $query = LancamentoContabil::query();

                // Aplicar os mesmos filtros da tabela
                $filters = request()->input('tableFilters', []);

                // Aplicar filtro de período
                if (isset($filters['data_emissao'])) {
                    if (isset($filters['data_emissao']['data_inicio'])) {
                        $query->whereDate('data_emissao', '>=', $filters['data_emissao']['data_inicio']);
                    }
                    if (isset($filters['data_emissao']['data_fim'])) {
                        $query->whereDate('data_emissao', '<=', $filters['data_emissao']['data_fim']);
                    }
                } else {
                    // Padrão: mês corrente
                    $query->whereMonth('data_emissao', now()->month)
                        ->whereYear('data_emissao', now()->year);
                }

                // Aplicar filtro de centro de custo
                if (isset($filters['centro_custo']['centro_custo_id'])) {
                    $query->where('centro_custo_id', $filters['centro_custo']['centro_custo_id']);
                }

                // Aplicar filtro de tipo
                if (isset($filters['tipo']['values']) && !empty($filters['tipo']['values'])) {
                    $query->whereIn('tipo', $filters['tipo']['values']);
                }

                // Aplicar filtro de status
                if (isset($filters['status']['values']) && !empty($filters['status']['values'])) {
                    $query->whereIn('status', $filters['status']['values']);
                }

                // Calcular totais
                $totalEntradas = (clone $query)->where('tipo', 'entrada')->sum('valor');
                $totalSaidas = (clone $query)->where('tipo', 'saida')->sum('valor');
                $saldo = $totalEntradas - $totalSaidas;

                // return view('filament.tables.headers.lancamento-summary', [
                //     'totalEntradas' => $totalEntradas,
                //     'totalSaidas' => $totalSaidas,
                //     'saldo' => $saldo,
                //     'totalGeral' => $totalEntradas + $totalSaidas,
                // ]);
            });
    }
}