<?php

namespace App\Filament\Resources\Fretes\Schemas;
use App\Models\Carga;
use App\Models\Cliente;
use App\Models\Frete;
use App\Models\Productos;
use Carbon\Carbon;
use Filament\Actions\Action;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Forms\Components\ToggleButtons;


class FreteForm
{
    public static function configure(Schema $schema): Schema
    {
       return $schema
            ->components([
        Grid::make(2)->schema([
                // ---------- CAMPOS DO FRETE ----------
            TextInput::make('numero_frete')
            ->default(function () {
        $date = now()->format('ymd'); // 250213
        $prefix = 'FRT-';

        // Consulta o último registro do dia atual
        $ultimo = Frete::whereDate('created_at', Carbon::now()->year)
                       ->orderBy('numero_frete', 'desc')
                       ->first();

        if ($ultimo) {
            // Extrai os últimos 4 caracteres do número
            $ultimoNumero = $ultimo->numero_frete;
            $sequencia = (int) substr($ultimoNumero, -4) + 1;
            $sequencia = str_pad($sequencia, 4, '0', STR_PAD_LEFT);
        } else {
            $sequencia = '0001';
        }

        return $prefix . $date . $sequencia;
    }),
            
            // VALOR DO FRETE (calculado automaticamente)
            TextInput::make('valor_frete')
                ->numeric()
                ->suffix('MT')
                ,
            
            // ... outros campos do frete (adiantamento, seguro, portagem, etc.)
            TextInput::make('valor_adiantamento')
                ->numeric()
                ->suffix('MT')
                ->reactive()
                
                ->afterStateUpdated(fn ($get, $set) => self::updateValorLiquido($get, $set)),
            
            TextInput::make('valor_liquido')
                ->numeric()
                ->suffix('MT')
                ,
                ]),
        Grid::make(2)->schema([
            TextInput::make('valor_seguro')
                ->numeric()
                ->suffix('MT')
                ->reactive()
                ->afterStateUpdated(fn (callable $get, callable $set) => self::updateValorLiquido($get, $set)),
            TextInput::make('valor_portagem')
                ->numeric()
                ->suffix('MT')
                ->reactive()
                ->afterStateUpdated(fn (callable $get, callable $set) => self::updateValorLiquido($get, $set))
                ,
            TextInput::make('valor_outros')
                ->numeric()
                ->suffix('MT')
                ->reactive()
                ->afterStateUpdated(fn (callable $get, callable $set) => self::updateValorLiquido($get, $set))
                ,

            ]),
                
            
            // ---------- REPEATER PARA TABELA PIVÔ (frete_carga) ----------
            Repeater::make('fretesCargas')
                ->label('Cargas deste Frete')
                ->relationship('fretesCargas') // HasMany para o modelo pivô
                ->schema([
                    
                    // --- SELECT com CRIAÇÃO DE CARGA VIA MODAL ---
                    Select::make('carga_id')
                        ->label('Carga')
                         ->options(fn () => Carga::all()->pluck('descricao', 'id')) // campo 'descricao' da tabela cargas
                        ->searchable()
                        ->required()
                        ->reactive()
                        ->createOptionForm([ // ⚡ MODAL PARA CRIAR NOVA CARGA
                            Grid::make(2)->schema([
                                Select::make('producto_id')
                                    ->label('Produto')
                                    ->options(fn () => Productos::pluck('nome', 'id')) 
                                    ->searchable()
                                    ->preload()
                                    ->required(),
                                Select::make('cliente_id')
                                    ->label('Cliente')
                                    ->options(fn () => Cliente::pluck('nome', 'id')) 
                                    ->searchable()
                                    ->preload()
                                    ->required(),
                                TextInput::make('descricao')
                                    ->label('Descrição da Carga')
                                    ->required()
                                    ->maxLength(255),
                                TextInput::make('peso')
                                    ->numeric()
                                    ->suffix('kg'),
                                TextInput::make('volume')
                                    ->numeric()
                                    ->suffix('m³'),
                                TextInput::make('tipo')
                                    ->maxLength(100),
                                TextInput::make('valor')
                                    ->label('Valor da Carga')
                                    ->numeric()
                                    ->suffix('MT')
                                    ->required(),
                            ])
                        ])
                        ->createOptionUsing(function (array $data): int {
                            // Salva a nova carga e retorna o ID
                            return Carga::create($data)->id;
                        })
                        ->createOptionAction(fn (Action $action) => $action->label('➕ Nova Carga')),
                    // ->createOptionButton('➕ Nova Carga'), // texto do botão
                    //     // ->createOptionForm('➕ Nova Carga'),
                    
                    // --- CAMPOS DA TABELA PIVÔ ---
                    TextInput::make('quantidade')
                        ->numeric()
                        ->default(1)
                        ->required()
                        ->reactive()
                        ->afterStateUpdated(fn ($get, $set) => self::calcularTotalLinha($get, $set)),
                    
                    TextInput::make('valor_unitario')
                        ->numeric()
                        ->required()
                        ->suffix('MT')
                        ->reactive()
                        ->afterStateUpdated(fn ($get, $set) => self::calcularTotalLinha($get, $set)),
                    
                    TextInput::make('valor_total')
                        ->numeric()
                        ->suffix('MT')
                        ,
                    
                    Textarea::make('observacao')
                        ->rows(1),
                    
                ])
                ->columns(5) // organize conforme seu layout
                ->columnSpan('full')
                ->createItemButtonLabel('➕ Adicionar Carga ao Frete')
                ->minItems(1)
                ->reactive()
                ->live() // ESSENCIAL para cálculos em tempo real
                ->afterStateUpdated(function (callable $get, callable $set) {
                    self::calcularValorFrete($get, $set);
                    self::updateValorLiquido($get, $set);
                }),
            ]);
    }

    // ---------- MÉTODOS AUXILIARES ----------

/**
 * Calcula o valor_total da linha (quantidade * valor_unitario)
 */
public static function calcularTotalLinha(callable $get, callable $set): void
{
    $qty = floatval($get('quantidade') ?? 1);
    $price = floatval($get('valor_unitario') ?? 0);
    $total = $qty * $price;
    $set('valor_total', $total);
    
    // Dispara recalculo do frete
    self::calcularValorFrete($get, $set);
}

/**
 * Soma todos os valor_total das linhas e atualiza o campo valor_frete
 */
public static function calcularValorFrete(callable $get, callable $set): void
{
    $itens = $get('fretesCargas') ?? [];
    $totalFrete = collect($itens)->sum('valor_total');
    $set('valor_frete', $totalFrete);
}

/**
 * Calcula o valor líquido (valor_frete - adiantamento - seguro - portagem - outros)
 */
public static function updateValorLiquido(callable $get, callable $set): void
{
    $valorFrete = floatval($get('valor_frete') ?? 0);

    $seguro = floatval($get('valor_seguro') ?? 0);
    $portagem = floatval($get('valor_portagem') ?? 0);
    $outros = floatval($get('valor_outros') ?? 0);
    
    $liquido = $valorFrete + $seguro + $portagem + $outros;
    $set('valor_liquido', $liquido);
}
}
